/*
 * Decompiled with CFR 0.152.
 */
package com.android.incfs.install;

import com.android.incfs.install.IBlockFilter;
import com.android.incfs.install.ILogger;
import com.android.incfs.install.PendingBlock;
import java.io.BufferedInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.channels.FileChannel;
import java.nio.channels.spi.AbstractInterruptibleChannel;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Base64;
import java.util.BitSet;
import java.util.List;
import java.util.Optional;
import org.gradle.internal.classpath.declarations.NioFileInterceptors;

class StreamingApk
implements AutoCloseable {
    static final short INCFS_BLOCK_SIZE = 4096;
    private static final int INCFS_DIGEST_SIZE = 32;
    private static final int INCFS_MAX_SIGNATURE_SIZE = 8096;
    private static final int INCFS_HASHES_PER_BLOCK = 128;
    private final Path mApk;
    private final Path mSignature;
    private final FileChannel mApkChannel;
    private final FileChannel mSignatureChannel;
    private final long mApkSize;
    private final long mSignatureSize;
    private final int mTreeOffset;
    private final String mSignatureBase64;
    private final int mDataBlockCount;
    private final int mTreeBlockCount;
    private final BitSet mSentDataBlocks;
    private final BitSet mSentTreeBlocks;
    private final IBlockFilter mBlockFilter;
    private final ILogger mLogger;

    private StreamingApk(Path apk, Path signature, FileChannel apkChannel, FileChannel signatureChannel, long apkSize, long sigSize, int treeOffset, String signatureBase64, IBlockFilter server, ILogger logger) {
        this.mApk = apk;
        this.mSignature = signature;
        this.mApkChannel = apkChannel;
        this.mSignatureChannel = signatureChannel;
        this.mApkSize = apkSize;
        this.mSignatureSize = sigSize;
        this.mDataBlockCount = StreamingApk.numBytesToNumBlocks(apkSize);
        this.mTreeBlockCount = StreamingApk.verityTreeBlocksForFile(apkSize);
        this.mSentDataBlocks = new BitSet(this.mDataBlockCount);
        this.mSentTreeBlocks = new BitSet(this.mTreeBlockCount);
        this.mTreeOffset = treeOffset;
        this.mSignatureBase64 = signatureBase64;
        this.mBlockFilter = server;
        this.mLogger = logger;
    }

    static StreamingApk generate(Path apk, Path sig, IBlockFilter server, ILogger logger) throws IOException {
        long sigSize;
        long apkSize;
        String base64Sig;
        int treeOffset;
        try (BufferedInputStream sigIs = new BufferedInputStream(NioFileInterceptors.intercept_newInputStream((Path)sig, (OpenOption[])new OpenOption[0], (String)"com.android.incfs.install.StreamingApk"));){
            int expectedTreeSize;
            ByteArrayOutputStream base64SigOS = new ByteArrayOutputStream();
            OutputStream sigBOS = Base64.getEncoder().wrap(base64SigOS);
            int version = StreamingApk.readInt32(sigIs, sigBOS, "Failed to read version from " + sig);
            int hashingInfoSize = StreamingApk.readBytesWithSize(sigIs, sigBOS, "Failed to read hashing info from " + sig);
            int signingInfoSize = StreamingApk.readBytesWithSize(sigIs, sigBOS, "Failed to read signing info from " + sig);
            int signatureSize = 12 + hashingInfoSize + signingInfoSize;
            if (signatureSize > 8096) {
                throw new IllegalArgumentException("Signature is too long. Max allowed is 8096");
            }
            int treeSize = StreamingApk.readInt32(sigIs, null, "Failed to read tree size from " + sig);
            if (treeSize != (expectedTreeSize = StreamingApk.verityTreeSizeForFile(Files.size(apk)))) {
                throw new IllegalArgumentException("Verity tree size mismatch in signature file: [was " + treeSize + ", expected " + expectedTreeSize + "]");
            }
            treeOffset = 4 + signatureSize;
            base64Sig = new String(base64SigOS.toByteArray());
        }
        FileChannel apkChannel = null;
        AbstractInterruptibleChannel signatureChannel = null;
        try {
            apkChannel = NioFileInterceptors.intercept_open((Path)apk, (OpenOption[])new OpenOption[0], (String)"com.android.incfs.install.StreamingApk");
            signatureChannel = NioFileInterceptors.intercept_open((Path)sig, (OpenOption[])new OpenOption[0], (String)"com.android.incfs.install.StreamingApk");
            apkSize = apkChannel.size();
            sigSize = ((FileChannel)signatureChannel).size();
        }
        catch (IOException e) {
            if (apkChannel != null) {
                apkChannel.close();
            }
            if (signatureChannel != null) {
                signatureChannel.close();
            }
            throw e;
        }
        return new StreamingApk(apk, sig, apkChannel, (FileChannel)signatureChannel, apkSize, sigSize, treeOffset, base64Sig, server, logger);
    }

    String getSignatureBase64() {
        return this.mSignatureBase64;
    }

    List<PendingBlock> getBlockResponse(int blockIndex) {
        if (blockIndex < 0 || blockIndex >= this.mDataBlockCount) {
            throw new IllegalArgumentException("Requested block index is outside range");
        }
        List<PendingBlock> responses = this.getTreeBlocksResponsesForDataBlock(blockIndex);
        if (!this.mSentDataBlocks.get(blockIndex)) {
            this.getDataPendingBlock(blockIndex).ifPresent(responses::add);
        }
        return responses;
    }

    private List<PendingBlock> getTreeBlocksResponsesForDataBlock(int blockIndex) {
        int dataBlockCount = this.mDataBlockCount;
        int totalNodeCount = this.mTreeBlockCount;
        int leafNodesCount = (dataBlockCount + 128 - 1) / 128;
        int leafNodesOffset = totalNodeCount - leafNodesCount;
        int leafIndex = leafNodesOffset + blockIndex / 128;
        ArrayList<PendingBlock> responses = new ArrayList<PendingBlock>();
        if (!this.mSentTreeBlocks.get(leafIndex)) {
            this.getTreePendingBlock(leafIndex).ifPresent(responses::add);
        }
        if (leafNodesOffset != 0) {
            for (int i = 0; i < leafNodesOffset; ++i) {
                if (this.mSentTreeBlocks.get(i)) continue;
                this.getTreePendingBlock(i).ifPresent(responses::add);
            }
        }
        return responses;
    }

    private Optional<PendingBlock> getTreePendingBlock(int treeBlockIndex) {
        int blockOffset = this.mTreeOffset + treeBlockIndex * 4096;
        short blockSize = (short)Math.min(this.mSignatureSize - (long)blockOffset, 4096L);
        PendingBlock response = new PendingBlock(this.mSignature, PendingBlock.Type.SIGNATURE_TREE, treeBlockIndex, this.mTreeBlockCount, this, blockOffset, blockSize);
        if (this.mBlockFilter.shouldServeBlock(response)) {
            this.mLogger.verbose("Sending %s", response);
            this.mSentTreeBlocks.set(treeBlockIndex, true);
            return Optional.of(response);
        }
        this.mLogger.verbose("Denied sending %s", response);
        return Optional.empty();
    }

    private Optional<PendingBlock> getDataPendingBlock(int index) {
        int blockOffset = index * 4096;
        short blockSize = (short)Math.min(this.mApkSize - (long)blockOffset, 4096L);
        PendingBlock response = new PendingBlock(this.mApk, PendingBlock.Type.APK_DATA, index, this.mDataBlockCount, this, blockOffset, blockSize);
        if (this.mBlockFilter.shouldServeBlock(response)) {
            this.mLogger.verbose("Sending %s", response);
            this.mSentDataBlocks.set(index, true);
            return Optional.of(response);
        }
        this.mLogger.verbose("Denied sending %s", response);
        return Optional.empty();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void readBlockData(ByteBuffer buffer, PendingBlock.Type type, int blockOffset, short blockSize) throws IOException {
        FileChannel channel = type == PendingBlock.Type.APK_DATA ? this.mApkChannel : this.mSignatureChannel;
        int previousLimit = buffer.limit();
        buffer.limit(buffer.position() + blockSize);
        try {
            if ((short)channel.read(buffer, blockOffset) != blockSize) {
                throw new IOException("Failed to read " + blockSize + " bytes from " + (type == PendingBlock.Type.APK_DATA ? this.mApk : this.mSignature));
            }
        }
        finally {
            buffer.limit(previousLimit);
        }
    }

    private static int numBytesToNumBlocks(long fileSize) {
        return (int)(fileSize / 4096L) + (fileSize % 4096L == 0L ? 0 : 1);
    }

    private static int verityTreeSizeForFile(long fileSize) {
        return StreamingApk.verityTreeBlocksForFile(fileSize) * 4096;
    }

    private static int verityTreeBlocksForFile(long fileSize) {
        if (fileSize == 0L) {
            return 0;
        }
        int hashPerBlock = 128;
        int totalTreeBlockCount = 0;
        long hashBlockCount = 1L + (fileSize - 1L) / 4096L;
        while (hashBlockCount > 1L) {
            hashBlockCount = (hashBlockCount + 128L - 1L) / 128L;
            totalTreeBlockCount = (int)((long)totalTreeBlockCount + hashBlockCount);
        }
        return totalTreeBlockCount;
    }

    private static int readInt32(InputStream is, OutputStream accumulator, String errorMessage) throws IOException {
        byte[] data = new byte[4];
        if (is.read(data) != 4) {
            throw new IOException(errorMessage);
        }
        if (accumulator != null) {
            accumulator.write(data);
        }
        return ByteBuffer.wrap(data).order(ByteOrder.LITTLE_ENDIAN).getInt();
    }

    private static int readBytesWithSize(InputStream is, OutputStream accumulator, String errorMessage) throws IOException {
        int totalRead;
        int length;
        int size = StreamingApk.readInt32(is, accumulator, errorMessage);
        byte[] buffer = new byte[4096];
        for (totalRead = 0; totalRead < size && (length = is.read(buffer, 0, size - totalRead)) > 0; totalRead += length) {
            accumulator.write(buffer, 0, length);
        }
        if (totalRead != size) {
            throw new IOException(errorMessage);
        }
        return size;
    }

    @Override
    public void close() {
        try {
            this.mApkChannel.close();
        }
        catch (IOException iOException) {
            // empty catch block
        }
        try {
            this.mSignatureChannel.close();
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }
}

